<?php

namespace App\Livewire;

use Livewire\Component;
use App\Models\Unduhan as PM;
use Livewire\Attributes\On;
use Livewire\WithFileUploads;
use Livewire\Attributes\Validate;


class Unduhan extends Component
{
    use WithFileUploads;
    public $judul, $aksi, $selectedId, $iter = 0;

    #[Validate('required')]
    #[Validate('mimes:png,jpg,pdf,docx,xls,xlsx,jpeg')]
    #[Validate('max:12288')]
    public $file;
    protected $messages = [
        '*.required' => 'Tidak Boleh Kosong',
        '*.file' => 'Harus File',
        '*.mimes' => 'File Harus Berformat :values',
        '*.max' => 'Maksimal :max kb'
    ];

    public function mount()
    {
        $this->aksi = "store";
    }

    public function store()
    {
        $this->validate(['judul' => 'required', 'file' => 'required|mimes:png,jpg,pdf,docx,xls,xlsx,jpeg|max:12288']);
        try {
            PM::create(['judul' => $this->judul, 'file' => $this->file->store('unduhan', 'public')]);
            $this->reset_fields();
            $this->success();
        } catch (\Throwable $th) {
            $this->error();
        }
    }

    public function edit($id)
    {
        $data = PM::find($id);
        $this->judul = $data->judul;
        $this->aksi = "update";
        $this->selectedId = $id;
    }

    public function update()
    {
        $this->validate(['judul' => 'required', 'file' => 'nullable|file|mimes:png,jpg,pdf,docx,xls,xlsx,jpeg']);
        try {
            $data = ['judul' => $this->judul];
            if ($this->file) {
                $data['file'] = $this->file->store('unduhan', 'public');
            }
            $lama = PM::find($this->selectedId);
            hapus_file($lama->file);
            $lama->update($data);
            $this->reset_fields();
            $this->success();
        } catch (\Throwable $th) {
            $this->error();
        }
    }

    #[On('destroy')]
    public function destroy($id)
    {
        try {
            $lama = PM::find($id);
            hapus_file($lama->file);
            $lama->delete();
            $this->reset_fields();
            $this->success();
        } catch (\Throwable $th) {
            $this->error();
        }
    }

    #[On('set-status')]
    public function set_status($id, $status)
    {
        try {
            PM::find($id)->update(['is_active' => $status]);
            $this->reset_fields();
            $this->success();
        } catch (\Throwable $th) {
            $this->error();
        }
    }

    public function reset_fields()
    {
        $this->reset(['judul', 'file', 'selectedId']);
        $this->resetValidation();
        $this->aksi = "store";
        $this->iter++;
    }

    public function success()
    {
        $this->dispatch('swal:modal', [
            'type' => 'success',
            'message' => 'Berhasil Simpan Data!',
            'toast' => 'true',
            'position' => 'top-end'
        ]);
    }

    public function error($msg = null)
    {
        $this->dispatch('swal:modal', [
            'type' => 'error',
            'message' => 'Operasi Gagal! ' . $msg,
            'toast' => 'true',
            'position' => 'top-end'
        ]);
    }

    public function confirm($id)
    {
        $this->dispatch('swal:confirm', [
            'type' => 'warning',
            'message' => 'Hapus Data?',
            'text' => 'Data Tidak Dapat Dikembalikan!',
            'id' => $id
        ]);
    }

    public function remove()
    {
        $this->dispatch('swal:modal', [
            'type' => 'success',
            'message' => 'Data Berhasil Dihapus!',
            'toast' => 'true',
            'position' => 'top-end'
        ]);
    }

    public function render()
    {
        $data = PM::all();
        return view('livewire.unduhan', compact('data'));
    }
}
