<?php

namespace App\Livewire\Pengaturan;

use Livewire\Component;
use Livewire\WithFileUploads;
use App\Models\Slider as PM;
use Illuminate\Support\Facades\Cache;
use Livewire\Attributes\On;
use Livewire\Attributes\Validate;
use Illuminate\Support\Facades\DB;

class Slider extends Component
{
    use WithFileUploads;
    public $iter = 0;
    #[Validate(['file.*' => 'required|image|mimes:png,jpg,jpeg|max:12288'])]
    public $file = [];
    protected $messages = [
        'file.*.required' => 'Tidak Boleh Kosong',
        'file.*.image' => 'Harus Gambar',
        'file.*.max' => 'Maksimal :max kb',
        'file.*.mimes' => 'Harus Berformat :values'
    ];

    public function store()
    {
        $this->validate();
        try {
            DB::beginTransaction();
            foreach($this->file as $file){
                PM::create(['file' => $file->store('slider', 'public')]);
            }
            DB::commit();
            Cache::forget('slider');
            $this->reset_fields();
            $this->success();
        } catch (\Throwable $th) {
            DB::rollBack();
            $this->error();
        }
    }

    #[On('destroy')]
    public function destroy($id)
    {
        try {
            $data = PM::find($id);
            hapus_file($data->file);
            $data->delete();
            $this->resetValidation();
            Cache::forget('slider');
            $this->success();
        } catch (\Throwable $th) {
            $this->error();
        }
    }

    #[On('set-status')]
    public function set_status($id, $status)
    {
        try {
            PM::find($id)->update(['is_active' => $status]);
            Cache::forget('slider');
            $this->success();
        } catch (\Throwable $th) {
            $this->error();
        }
    }

    public function reset_fields()
    {
        $this->reset('file');
        $this->resetValidation();
        $this->iter++;
    }

    public function success()
    {
        $this->dispatch('swal:modal', [
            'type' => 'success',
            'message' => 'Berhasil Simpan Data!',
            'toast' => 'true',
            'position' => 'top-end'
        ]);
    }

    public function error($msg = null)
    {
        $this->dispatch('swal:modal', [
            'type' => 'error',
            'message' => 'Operasi Gagal! ' . $msg,
            'toast' => 'true',
            'position' => 'top-end'
        ]);
    }

    public function confirm($id)
    {
        $this->dispatch('swal:confirm', [
            'type' => 'warning',
            'message' => 'Hapus Data?',
            'text' => 'Data Tidak Dapat Dikembalikan!',
            'id' => $id
        ]);
    }

    public function remove()
    {
        $this->dispatch('swal:modal', [
            'type' => 'success',
            'message' => 'Data Berhasil Dihapus!',
            'toast' => 'true',
            'position' => 'top-end'
        ]);
    }

    public function render()
    {
        $data = PM::orderBy('created_at')->get();
        return view('livewire.pengaturan.slider', compact('data'));
    }
}
