<?php

namespace App\Livewire\Pengaturan;

use Livewire\Component;
use App\Models\MediaSosial as PM;
use Livewire\Attributes\On;
use Illuminate\Support\Str;

class MediaSosial extends Component
{
    public $url, $platform, $aksi, $selectedId;
    protected $messages = [
        '*.required' => 'Tidak Boleh Kosong',
        '*.unique' => 'Data Sudah Ada'
    ];

    public function mount()
    {
        $this->aksi = "store";
    }

    public function store()
    {
        $this->validate(['platform' => 'required|unique:media_sosial,platform', 'url' => 'required']);
        try {
            PM::create(['platform' => $this->platform, 'url' => $this->url]);
            $this->reset_fields();
            $this->success();
        } catch (\Throwable $th) {
            $this->error();
        }
    }

    public function edit(PM $data)
    {
        $this->platform = $data->platform;
        $this->url = $data->url;
        $this->aksi = "update";
        $this->selectedId = $data->id;
    }

    public function update()
    {
        $this->validate(['platform' => 'required|unique:media_sosial,platform,'.$this->selectedId, 'url' => 'required']);
        try {
            PM::find($this->selectedId)->update(['platform' => $this->platform, 'url' => $this->url]);
            $this->reset_fields();
            $this->success();
        } catch (\Throwable $th) {
            $this->error();
        }
    }

    #[On('destroy')]
    public function destroy($id)
    {
        try {
            PM::find($id)->delete();
            $this->reset_fields();
            $this->success();
        } catch (\Throwable $th) {
            $this->error();
        }
    }

    public function reset_fields()
    {
        $this->reset(['platform', 'url', 'selectedId']);
        $this->resetValidation();
        $this->aksi = "store";
    }

    public function success()
    {
        $this->dispatch('swal:modal', [
            'type' => 'success',
            'message' => 'Berhasil Simpan Data!',
            'toast' => 'true',
            'position' => 'top-end'
        ]);
    }

    public function error($msg = null)
    {
        $this->dispatch('swal:modal', [
            'type' => 'error',
            'message' => 'Operasi Gagal! ' . $msg,
            'toast' => 'true',
            'position' => 'top-end'
        ]);
    }

    public function confirm($id)
    {
        $this->dispatch('swal:confirm', [
            'type' => 'warning',
            'message' => 'Hapus Data?',
            'text' => 'Data Tidak Dapat Dikembalikan!',
            'id' => $id
        ]);
    }

    public function remove()
    {
        $this->dispatch('swal:modal', [
            'type' => 'success',
            'message' => 'Data Berhasil Dihapus!',
            'toast' => 'true',
            'position' => 'top-end'
        ]);
    }

    public function render()
    {
        $data = PM::all();
        $data->map(function ($item) {
            if ($item->platform == 'whatsapp') {
                $url = Str::substr($item->url, 0, 1) == '0' ? Str::replaceStart('0', '62', $item->url) : $item->url;
                $item->url = 'https://wa.me/' . $url;
            }
            if ($item->platform == 'email') {
                $item->url = 'mailto:' . $item->url;
            }

            return $item;
        });
        return view('livewire.pengaturan.media-sosial', compact('data'));
    }
}
