<?php

namespace App\Livewire;
use Livewire\Attributes\On;
use Illuminate\Support\Str;
use App\Models\AlbumKegiatan as PM;
use App\Models\FotoKegiatan as FM;
use Livewire\WithPagination;
use Livewire\WithoutUrlPagination;

use Livewire\Component;

class AlbumKegiatan extends Component
{
    use WithPagination, WithoutUrlPagination;
    protected $paginationTheme = 'bootstrap';
    public $judul, $deskripsi, $tanggalPublish, $statusPublish, $aksi, $selectedId, $iter = 1;
    protected $messages = [
        '*.required' => 'Tidak Boleh Kosong',
    ];

    public function mount()
    {
        $this->aksi = "store";
    }

    public function store()
    {
        $this->validate(['judul' => 'required', 'deskripsi' => 'nullable', 'tanggalPublish' => 'required|date', 'statusPublish' => 'required']);
        try {
            PM::create(['judul' => $this->judul, 'slug' => $this->unique_slug(Str::slug($this->judul)), 'deskripsi' => $this->deskripsi != "" ? $this->deskripsi : null, 'tanggal_publish' => $this->tanggalPublish, 'is_active' => $this->statusPublish]);
            $this->reset_fields();
            $this->success();
        } catch (\Throwable $th) {
            $this->error();
        }
    }

    public function edit($id)
    {
        $data = PM::find($id);
        $this->judul = $data->judul;
        $this->deskripsi = $data->deskripsi;
        $this->tanggalPublish = $data->tanggal_publish;
        $this->statusPublish = $data->is_active;
        $this->aksi = "update";
        $this->selectedId = $id;
    }

    public function update()
    {
        $this->validate(['judul' => 'required', 'deskripsi' => 'nullable', 'tanggalPublish' => 'required|date', 'statusPublish' => 'required']);
        try {
            PM::find($this->selectedId)->update(['judul' => $this->judul, 'slug' => $this->unique_slug(Str::slug($this->judul)), 'deskripsi' => $this->deskripsi != "" ? $this->deskripsi : null, 'tanggal_publish' => $this->tanggalPublish, 'is_active' => $this->statusPublish]);
            $this->reset_fields();
            $this->success();
        } catch (\Throwable $th) {
            $this->error();
        }
    }

    #[On('destroy')]
    public function destroy($id)
    {
        try {
            $album = PM::find($id);
            $album->to_foto_kegiatan->map(function($item){
                hapus_file($item->file);
            });
            $album->delete();
            $this->reset_fields();
            $this->success();
        } catch (\Throwable $th) {
            $this->error();
        }
    }

    public function reset_fields()
    {
        $this->reset(['judul', 'deskripsi', 'tanggalPublish', 'statusPublish', 'selectedId']);
        $this->resetValidation();
        $this->aksi = "store";
        $this->iter++;
        $this->resetPage();
    }

    public function unique_slug($slug)
    {
        $isExist = true;
        $i = 1;
        do {
            if ($this->check_slug($slug) == 0) {
                $isExist = false;
            } else {
                $slug = $slug . '-' . $i;
            }
            $i++;
        } while ($isExist);
        return $slug;
    }

    public function check_slug($slug)
    {
        return PM::where('slug', $slug)->count();
    }

    public function success()
    {
        $this->dispatch('swal:modal', [
            'type' => 'success',
            'message' => 'Berhasil Simpan Data!',
            'toast' => 'true',
            'position' => 'top-end'
        ]);
    }

    public function error($msg = null)
    {
        $this->dispatch('swal:modal', [
            'type' => 'error',
            'message' => 'Operasi Gagal! ' . $msg,
            'toast' => 'true',
            'position' => 'top-end'
        ]);
    }

    public function confirm($id)
    {
        $this->dispatch('swal:confirm', [
            'type' => 'warning',
            'message' => 'Hapus Data?',
            'text' => 'Data Tidak Dapat Dikembalikan!',
            'id' => $id
        ]);
    }

    public function remove()
    {
        $this->dispatch('swal:modal', [
            'type' => 'success',
            'message' => 'Data Berhasil Dihapus!',
            'toast' => 'true',
            'position' => 'top-end'
        ]);
    }

    public function render()
    {
        $data = PM::orderBy('tanggal_publish')->paginate(10);
        return view('livewire.album-kegiatan', compact('data'));
    }
}
